<?php
/******************************************************************************
** codebase.php
*  Coppermine 1.6.x Plugin - suppressor. Created by KF to suppress display of file information
*                   Updated May 2024 to Version 1.1 to suppress title information when hovering 
*                   mouse pointer on thumbnails.
*                   Version 1.2 add feature to i) display or suppress uploader name based on gallery keyword
*                   ii) disable display of EXIF information and config-defined custom image description information 
*                   on the Image Display screen if NOEXIF keyword defined for the album
*                   Version 1.2.1 fixes a limitation of 1.2 where, despite suppressing uploader name, the uploader name
*                   could be displayed with the thumbs by, for example, choosing the "Last Uploads" menu option.
*                   Other similar limitations have been addressed. 
*  Version 1.2.1
* @since  1.6.22
******************************************************************************/

/******************************************************************************
 * The code here gets awkward because of languages. The Configuration is all in English and the config screen can be changed 
 * in the language file for the plugin. Picking up languages for the config from language files elsewhere 
 * in Coppermine would need custom language code in the plugin to manipulate the files and is not feasible.
 * 
 * In the running of the plugin, language files are available and have to be used because the items of picture info are
 * named in the chosen language. The plugin settings are all coded using the English names and pic info has to be checked against
 * the chosen language in suppressor_filter_pic_info_item.  Menu items do not have localised names in the code where the checking is done.
 * 
******************************************************************************/

// Make sure that this file can't be accessed directly, but only from within the Coppermine user interface
if (!defined('IN_COPPERMINE')) {
    die('Not in Coppermine...');
}

// Define plugin actions here
$thisplugin->add_action('plugin_install', 'suppressor_install');
$thisplugin->add_action('plugin_uninstall', 'suppressor_uninstall');
$thisplugin->add_action('plugin_cleanup', 'suppressor_cleanup');
$thisplugin->add_action('post_breadcrumb', 'suppressor_check_uploader');

// Define plugin filters here
$thisplugin->add_filter('file_info', 'suppressor_filter_pic_info');
$thisplugin->add_filter('html_image', 'suppressor_filter_image_html');
$thisplugin->add_filter('sub_menu', 'suppressor_filter_menu');
$thisplugin->add_filter('theme_display_thumbnails_params', 'suppressor_thumb_title');
$thisplugin->add_filter('thumb_caption_topn', 'suppressor_filter_thumb_captions');
$thisplugin->add_filter('thumb_caption', 'suppressor_filter_thumb_captions');
$thisplugin->add_filter('thumb_caption_lastup', 'suppressor_filter_thumb_captions');
$thisplugin->add_filter('thumb_caption_lastupby', 'suppressor_filter_thumb_pic');
$thisplugin->add_filter('thumb_data', 'suppressor_filter_thumb');


function suppressor_init_settings(&$suppressor_setting) {
    // These are the indexes into the configuration string for the settings, starting from 1
    $suppressor_setting['Filename']            = 1;
    $suppressor_setting['Filesize']            = 2;
    $suppressor_setting['Date_added']          = 3;
    $suppressor_setting['Dimensions']          = 4;
    $suppressor_setting['Displayed']           = 5;
    $suppressor_setting['DateTime_Original']   = 6;
    $suppressor_setting['URL']                 = 7;
    $suppressor_setting['Favorites']           = 8;
    $suppressor_setting['Search']              = 9;
    $suppressor_setting['Top_rated']           = 10;
    $suppressor_setting['Most_viewed']         = 11;
    $suppressor_setting['Last_Comments']       = 12;
    $suppressor_setting['Last_Uploads']        = 13;
    $suppressor_setting['Keywords']            = 14;
    return true;
} 

function suppressor_filter_pic_info_item(&$file_info, $suppressor_setting, $item_name, $screen_name) {
    // Need to use the local language version ($screen_name) for the check as this is used in displayimage.php
    global $CONFIG;
    $key_string = $screen_name;
    if (substr($CONFIG['suppressor_settings'], ($suppressor_setting[$item_name] - 1), 1) == "0") {   // Check if config setting (substr counts from zero) is 0
        $file_info = array_diff($file_info, array($file_info[$key_string]));          // If it is, remove item of file info from the array of retrieved info 
        return true;
    } else {
        return false;
    }
}

function suppressor_filter_pic_info($file_info) {
    global $lang_common, $lang_picinfo, $CONFIG, $CURRENT_ALBUM_DATA, $CPG_PHP_SELF, $CURRENT_PIC_DATA;
    suppressor_init_settings($file_info_setting);
    suppressor_filter_pic_info_item($file_info, $file_info_setting, 'Filename',          $lang_common['filename'] );
    suppressor_filter_pic_info_item($file_info, $file_info_setting, 'Filesize',          $lang_common['filesize'] );
    suppressor_filter_pic_info_item($file_info, $file_info_setting, 'Date_added',        $lang_picinfo['Date Added'] );
    suppressor_filter_pic_info_item($file_info, $file_info_setting, 'Dimensions',        $lang_picinfo['Dimensions'] );
    suppressor_filter_pic_info_item($file_info, $file_info_setting, 'Displayed',         $lang_picinfo['Displayed'] );
    suppressor_filter_pic_info_item($file_info, $file_info_setting, 'DateTime_Original', $lang_picinfo['DateTimeOriginal'] );
    suppressor_filter_pic_info_item($file_info, $file_info_setting, 'URL',               $lang_picinfo['URL'] );
    suppressor_filter_pic_info_item($file_info, $file_info_setting, 'Favorites',         $lang_picinfo['addFavPhrase']);
    suppressor_filter_pic_info_item($file_info, $file_info_setting, 'Keywords',          $lang_common['keywords'] );
    // By default displayimage.php displays the name of the uploader. Check, and if we don't want it displayed, execute the clause below
    // At this point in execution displayimage.php has not set the value of $CURRENT_ALBUM_KEYWORD so we have to extract the keyword ourselves
    // Album keyword contains UPLOADER if we want to force display of uploader name in thumbnails.php or NOUPLOADER if we don't want it. 
    // The default in displayimage.php is to always display uploader name so UPLOADER in the keyword has no effect here
    $alb_info = get_album_name($CURRENT_ALBUM_DATA['aid']);     // Despite its name, function returns album name AND keyword 
    if ($alb_info['keyword'] != '') {
        $upl_keyword_found = stripos($alb_info['keyword'], "NOUPLOADER");
        if (is_integer($upl_keyword_found) && !GALLERY_ADMIN_MODE) {   
        // We want uploader name display to be disabled for this album so overwrite field with album name
            $file_info[$lang_picinfo['Album name']] = $alb_info['title'];
        }
        if ($CPG_PHP_SELF == "displayimage.php") {
            $exif_keyword_found = stripos($alb_info['keyword'], "NOEXIF");  // Look for the NOEXIF keyword which suppresses EXIF display
            if (is_integer($exif_keyword_found)) {                          // Also delete the custom image-description fields if the keyword is set
                if (isset($CURRENT_PIC_DATA['user1'])) unset ($file_info[$CONFIG['user_field1_name']]);
                if (isset($CURRENT_PIC_DATA['user2'])) unset ($file_info[$CONFIG['user_field2_name']]);
                if (isset($CURRENT_PIC_DATA['user3'])) unset ($file_info[$CONFIG['user_field3_name']]);
                if (isset($CURRENT_PIC_DATA['user4'])) unset ($file_info[$CONFIG['user_field4_name']]);
            }
        }
    }
    return $file_info;
}

function suppressor_filter_image_html($pic_html) {
    global $CONFIG;
    suppressor_init_settings($info_setting);
    if (substr($CONFIG['suppressor_settings'], ($info_setting['Keywords'] - 1), 1) == "0") {   // If Keywords are configured to be suppressed
        // Search through HTML starting from Keywords: up to and excluding non-alpha numeric character and replace with nothing
        $pic_html = preg_replace('/Keywords: [[:alnum:]]+/', '', $pic_html);   
    }
    suppressor_check_exif();    
    return $pic_html;
}

function suppressor_filter_menu_item(&$menu_changed, $file_info_to_check, &$buttonset, $menu_link) {
    global $CONFIG;
    if (substr($CONFIG['suppressor_settings'], ($file_info_to_check - 1), 1) == "0") {   // If this is a menu item configured to be suppressed
        $button_count = 0;
        foreach ($buttonset as $button) {                                   // Loop through the buttons. Include all except the favorites menu
            if ($button[0] <> $menu_link) {
                $buttons_out[$button_count] = $button;
                $button_count++;
            } 
        }
        $buttonset = $buttons_out;  // Return with revised buttons / links
        $menu_changed = true;
    } 
}

function suppressor_filter_menu($buttonset) {  
    suppressor_init_settings($file_info_setting);
    $menu_changed = false;
    suppressor_filter_menu_item($menu_changed, $file_info_setting['Favorites'],     $buttonset, '{FAV_LNK}');
    suppressor_filter_menu_item($menu_changed, $file_info_setting['Search'],        $buttonset, '{SEARCH_LNK}');
    suppressor_filter_menu_item($menu_changed, $file_info_setting['Top_rated'],     $buttonset, '{TOPRATED_LNK}');
    suppressor_filter_menu_item($menu_changed, $file_info_setting['Most_viewed'],   $buttonset, '{TOPN_LNK}');
    suppressor_filter_menu_item($menu_changed, $file_info_setting['Last_Comments'], $buttonset, '{LASTCOM_LNK}');
    suppressor_filter_menu_item($menu_changed, $file_info_setting['Last_Uploads'],  $buttonset, '{LASTUP_LNK}');
    // If we changed anything, need to be sure that the last item in the array of menu items has the caret divider between items set to blank
    if ($menu_changed) {
        $last_item_index = count($buttonset) - 1;    //Noting that array starts from zero
        $buttonset[$last_item_index][4] = '';        //Blank the caret
        $buttonset[$last_item_index][5] = '';   
    }
    return($buttonset); 
}

function suppressor_filter_title_item(&$paramstring, $suppressor_setting, $regex_string) {
    global $CONFIG;
    if (substr($CONFIG['suppressor_settings'], ($suppressor_setting - 1), 1) == "0") {   // Check if config setting (substr counts from zero) is 0
        $paramstring = preg_replace($regex_string, '', $paramstring );      // If it is, do the regex match and replace with blank string (delete)
        return true;
    } else {
        return false;
    }
}

function suppressor_thumb_title($params) {
    suppressor_init_settings($suppressor_setting);
    $filter_filename = suppressor_filter_title_item($params['{THUMB}'], $suppressor_setting['Filename'],   '/Filename=.*\n/m');  // Check filename item
    $filter_filesize = suppressor_filter_title_item($params['{THUMB}'], $suppressor_setting['Filesize'],   '/Filesize=.*\n/m');  // Check filesize item
    $filter_dimensions = suppressor_filter_title_item($params['{THUMB}'], $suppressor_setting['Dimensions'], '/Dimensions=.*\n/m');  // Check dimensions item
    $filter_date_added = suppressor_filter_title_item($params['{THUMB}'], $suppressor_setting['Date_added'], '/Date added=.*[[:alnum:]]+/');  // Check date added item
    if ($filter_filename && $filter_filesize && $filter_dimensions && $filter_date_added ) {
        $params['{THUMB}'] = preg_replace('/title=.*"/', '', $params['{THUMB}']);          // Delete the title="" ...   string and keep the end of the string:  />
    }
    return($params);
}

function suppressor_filter_thumb_captions($rowset) {
    // Function is called to go through the captions one by one for all the thumbs in the lastup and topn meta albums and filter out uploader name
    foreach ($rowset as &$row) {
        $alb_info = get_album_name($row['aid']);     // Despite its name, function returns album name AND keyword 
        if ($alb_info['keyword'] != '') {
            $upl_keyword_found = stripos($alb_info['keyword'], "NOUPLOADER");
            if (is_integer($upl_keyword_found) && !GALLERY_ADMIN_MODE) {                  // Test if NOUPLOADER keyword is found and not Admin mode
                $pattern  = '/' . 'thumb_title_owner">.*?<\/span>' . '/i';     // Text to look for - must be enclosed with delimiter such as /
                $repltext = 'thumb_title_owner"><a>(------)</a></span>';       // Replacement text 
                $new_text = preg_replace($pattern, $repltext, $row['caption_text']);
                $row['caption_text'] = $new_text;
            }
        }
    }
    return($rowset);
}

function suppressor_filter_thumb_pic($rowset) {
    // Called when thumbs are displayed by meta-album lastupby. Need the thumb itself hidden if its home album is set to NOUPLOADER and not Admin mode
    global $lang_plugin_suppressor;
    foreach ($rowset as &$row) {
        $alb_info = get_album_name($row['aid']);     // Despite its name, function returns album name AND keyword 
        if ($alb_info['keyword'] != '') {
            $upl_keyword_found = stripos($alb_info['keyword'], "NOUPLOADER");
            if (is_integer($upl_keyword_found) && !GALLERY_ADMIN_MODE) {                  // Test if NOUPLOADER keyword is found and it's not Admin mode
                // If it is, overwrite uploader name with symbols
                $pattern  = '/' . 'thumb_title_owner">.*?<\/span>' . '/i';     // Text to look for - must be enclosed with delimiter such as /
                $repltext = 'thumb_title_owner"><a>(------)</a></span>';       // Replacement text 
                $new_text = preg_replace($pattern, $repltext, $row['caption_text']);
                $row['caption_text'] = $new_text;
                $row['filepath'] = 'plugins/suppressor/icons/';
                $row['filename'] = 'private.jpg';
                $row['url_prefix'] = 'plugins/suppressor/icons/';
                $pattern  = '/' . 'thumb_title_title">.*?<\/span>' . '/i';     // Text to look for up to the first </span>- must be enclosed with delimiter such as /
                $repltext = 'thumb_title_title">(------)</span>';   // Replacement text which must include the end of the string 
                $new_text = preg_replace($pattern, $repltext, $row['caption_text']);
                $row['caption_text'] = $new_text;
                $pattern  = '/' . 'thumb_caption_ctime">.*?<\/span>' . '/i';    // Text to look for up to the first </span>- must be enclosed with delimiter such as /
                $repltext = 'thumb_caption_ctime">(------)</span>';             // Replacement text which must include the end of the string 
                $new_text = preg_replace($pattern, $repltext, $row['caption_text']);
                $row['caption_text'] = $new_text;
            }
        }
    }
    return($rowset);

}

function suppressor_filter_thumb($rowset) {
    // Suppression of the thumbnail image itself is only wanted when displaying a user's profile. In all other calls drop out
    global $CONFIG, $CPG_PHP_SELF;
    if ($CPG_PHP_SELF == "profile.php") {
        $fname = $rowset['filename'];
        $result = cpg_db_query("SELECT aid FROM {$CONFIG['TABLE_PICTURES']} WHERE filename = '$fname'");
        $row = $result->fetchAssoc();
        $thumb_aid = $row['aid'];
        $alb_info = get_album_name($thumb_aid);     // Despite its name, function returns album name AND keyword 
        if ($alb_info['keyword'] != '') {
            $noupl_keyword_found = stripos($alb_info['keyword'], "NOUPLOADER");  // Look for the NOUPLOADER keyword which suppresses display of uploader name 
            if (is_integer($noupl_keyword_found)  && !GALLERY_ADMIN_MODE) {
                $rowset['filename'] = 'thumb_private.jpg';
                $rowset['filepath'] = 'plugins/suppressor/icons/';
                $rowset['extension'] = 'png';
                $rowset['url']      = $rowset['filepath'] . $rowset['filename'];
            }
        }
    }
    return($rowset);
}

function suppressor_check_uploader() {
    // Check if the gallery keyword contains string NOUPLOADER or UPLOADER meaning the configured (yes/no) display of uploader name should be 
    // overriden for this album. The $config setting is used by the code that displays thumbnails. It's not used for the picture information display
    global $CONFIG, $CURRENT_ALBUM_KEYWORD, $CURRENT_ALBUM_DATA, $CPG_PHP_SELF, $CURRENT_PIC_DATA;
    if ($CPG_PHP_SELF == "thumbnails.php") {
        $album_keyword_upl = "UPLOADER";
        $album_keyword_noupl = "NOUPLOADER";
        if ($CURRENT_ALBUM_KEYWORD != '') {
            $keyword_found = stripos($CURRENT_ALBUM_KEYWORD,$album_keyword_noupl);  // if keyword is found an integer type is returned
            if (is_integer($keyword_found) && !GALLERY_ADMIN_MODE) {   // We want uploader name display to be disabled for this album
                $CONFIG['display_uploader'] = 0;
            } else {
                $keyword_found = stripos($CURRENT_ALBUM_KEYWORD,$album_keyword_upl);
                if (is_integer($keyword_found)) {
                    $CONFIG['display_uploader'] = 1;
                }
            }
        }
    }
    return true;
}

function suppressor_check_exif() {
    // Check if the gallery keyword contains string NOEXIF  meaning the configured (yes/no) display of EXIF information should be 
    // overriden for this album. The $config setting is changed in memory only and prevents function html_picinfo calling exif_parse_file to
    // insert the EXIF information
    global $CONFIG, $CURRENT_ALBUM_DATA, $CPG_PHP_SELF;
    if ($CPG_PHP_SELF == "displayimage.php") {
        if ($CONFIG['read_exif_data'] == 1) {                           // If EXIF information is being read from the image
            $alb_info = get_album_name($CURRENT_ALBUM_DATA['aid']);     // Despite its name, function returns album name AND keyword 
            if ($alb_info['keyword'] != '') {
                $exif_keyword_found = stripos($alb_info['keyword'], "NOEXIF");  // Look for the NOEXIF keyword which suppresses EXIF display
                if (is_integer($exif_keyword_found)) {   
                    $CONFIG['read_exif_data'] = 0;                      // Change the flag (in memory only) so EXIF isn't read
                }
            }
        }
    }
    return true;
}

function suppressor_install() {
    global $CONFIG, $suppressor_default; 

    cpg_db_query("INSERT IGNORE INTO {$CONFIG['TABLE_CONFIG']} (`name`, `value`) 
                 VALUES ('suppressor_settings', {$suppressor_default})
                ");
    return true;
}

function suppressor_uninstall() {
    global $CONFIG, $lang_errors;
    $superCage = Inspekt::makeSuperCage();
    if (!checkFormToken()) {
        cpg_die(ERROR, $lang_errors['invalid_form_token'], __FILE__, __LINE__);
    }
	if ($superCage->post->keyExists('suppressor_submit') && $superCage->post->keyExists('suppressor_uninstall_type')) {
		if ($superCage->post->getInt('suppressor_uninstall_type') == 1) {
            // Delete the plugin config records
            cpg_db_query("DELETE FROM {$CONFIG['TABLE_CONFIG']} WHERE name IN ( 
                        'suppressor_settings' 
                       )");
        }
        return true;
    } else {
        return 1;
    }
}

function suppressor_cleanup ()  {
	global $lang_common, $superCage, $lang_plugin_suppressor;

	echo '<form name="cpgform" id="cpgform" action="'.$superCage->server->getEscaped('REQUEST_URI').'" method="post">';
	echo <<<EOT
	<br />{$lang_plugin_suppressor['uninstall_type']}<br />
	<input type="radio" name="suppressor_uninstall_type" id="suppressor_full" value="1" checked="checked" /> 
        <label for="suppressor_full">{$lang_plugin_suppressor['uninstall_full']}</label><br />
	<input type="radio" name="suppressor_uninstall_type" id="suppressor_partial" value="2"  /> 
        <label for="suppressor_partial">{$lang_plugin_suppressor['uninstall_partial']}</label><br />
	<br /><input type="submit" name="suppressor_submit" value="{$lang_common['continue']}" class="button" style="padding-right:5px;padding-left:5px" />
	</form>
EOT;
}
